%======================================================%
%====== USING EYELINK WITH PSYCHTOOLBOX IN MATLAB =====%
%====== Guide to accompany tutuorial for the BPRG =====%
%======   James Bartolotti and Scott Schroeder    =====%
%======      j-bartolotti@u.northwestern.edu      =====%
%======       schroeder@u.northwestern.edu        =====%
%======                06/01/2016                 =====%
%======================================================%
%
% For additional information, refer to the EyelinkToolbox documentation:
% http://docs.psychtoolbox.org/EyelinkToolbox

%NOTE: In this tutorial, commands that you need to change for your own experiment are marked by a comment with three asterisks, ***


%=========================================#
%====== STEP 0: Install Psychtoolbox =====#
%=========================================#

%Eyelink support in Matlab is enabled through Psychtoolbox-3.
%Installation instructions vary by system, go to http://psychtoolbox.org/download/ for a full walkthrough.

%Installation typically requires first downloading a Subversion utility, and second running a DownloadPsychtoolbox command in Matlab.

function myExperiment()
	%% EYETRACKER DEBUGGING 

	% Defines a "Dummymode" flag that you can set to 1 (true) or 0 (false) to determine whether to use the eyetracker or not. This is very useful for testing and debugging on a different computer than the eyetracker-computer.
	settings.eyetracker.dummy_mode = 0;
	eyelink_success = Eyelink('initialize'); %true if the eyetracker is detected, false if initialization fails
		

	if eyelink_success
		if settings.eyetracker.dummy_mode == 1
			warndlg("WARNING: Dummy mode enabled. Not recording from the Eyetracker!") %warning when the eyetracker is connected, but you are not recording from it (dummy mode)
		end
	else
		if settings.eyetracker.dummy_mode == 0
			warndlg("WARNING: Eyetracker not detected, and dummy mode is OFF. Connect the eyetracker, or  set settings.eyetracker.dummy_mode to false.") %warning when you want to use the eyetracker (i.e., dummy mode is off), but the eyetracker is not connected 
		end
	end


	%% EXPERIMENT INITIALIZATION

	% Your experiment-specific settings go here, before the main loop of your program
	number_of_trials = 10;
	
	% pops up an input dialog in which to type this participant's code. We specify a five character max, because we use three more characters for the experimenter code and study code and the eyetracker's EDF files only allows eight characters in the filename.
	id=inputdlg('Subject ID? (Max 5 characters)');
	subject_id=id{1};
	clear id
	
	experimenter_code = 'JB'; %*** set to your personal code
	study_code = 'A'; %*** set to this study's code
	subject_code = num2str(subject_id);

	edf_filename = [experimenter_code, study_code, subject_code]; % JBA10, for subject 10

	base_folder = fullfile('~/MATLAB'); %*** Set this to the base matlab directory
	experimenter_folder = 'James'; %*** Set to your personal folder
	study_folder = 'StudyA'; %*** set to this study's folder within your personal folder
	eyetracking_data_folder = 'EDF_files';
	save_folder = fullfile(base_folder, experimenter_folder, study_folder, eyetracking_data_folder);

	%Pyschtoolbox-related setup
	number_of_screens=max(Screen('Screens'));
	main_window=Screen('OpenWindow', number_of_screens); %open the highest-numbered screen, which should usually be the right one in a multi-display setup.
	
	do_eyetracker_setup(settings.eyetracker.dummy_mode, edf_filename, main_window); %sub-function that runs through the eyetracker setup routine

	
	%% MAIN EXPERIMENT
	%***In your experiment you will want to load stimuli information and target/competitor/filler locations from an external text file
	number_of_trials = 2;
	target_pictures = {'apple_picture_file', 'banana_picture_file'};
	target_locations = [1,2];
	competitor_pictures = {'cat_picture_file', 'dog_picture_file'};
	competitor_locations = [2,3];
	filler1_pictures = {'ear_picture_file', 'foot_picture_file'};
	filler1_locations = [3,4];
	filler2_pictures = {'green_picture_file', 'horange_picture_file'};
	filler2_locations = [4,1];

	%These vectors will hold accuracy and response time data for each trial. We initialize them as vectors full of null data (NaN)
	accuracy = NaN(1,number_of_trials);
	rt = NaN(1,number_of_trials);
	
	for i = 1:number_of_trials %*** You will want to randomize trial presentation or use a pseudorandom order.
		trialdata.target.picture = target_pictures{i};
		trialdata.target.location = target_locations[i];
		trialdata.competitor.picture = competitor_pictures{i};
		trialdata.target.location = competitor_locations[i];
		trialdata.filler1.picture = filler1_pictures{i};
		trialdata.target.location = filler1_locations[i];
		trialdata.filler2.picture = filler2_pictures{i};
		trialdata.target.location = filler2_locations[i];
		
		Eyelink('StartRecording'); %start recording eye position
		WaitSecs(0.1); %Get a few samples before we actually start the trial to account for eyetracker startup fluctuations. This time will get wrapped into your inter-trial-interval.
		Eyelink('Message', 'SYNCTIME'); %Mark the zero-plot time in the eyelink data file

		[accuracy(i), rt(i)] = RunTrial(trialdata); %***In your experiment, you will want to pass information about the current trial to your RunTrial function. Accuracy and RT data get returned from the function after each trial and put into the vectors. Best practice is to save these as .mat or .txt files after each trial so that you don't lose data in the event of a computer crash.
		
		Eyelink('StopRecording'); % Stop recording at the end of each trial.
		
		%If your study is long, you may want to periodically reset the eyetracker calibration, using the following commands.
		%el = EyelinkInitDefaults(main_window) 
		%EyelinkDoTrackerSetup(el); % Calibrate the eye tracker
		%EyelinkDoDriftCorrection(el); 	% Check the calibration accuracy

	end
	
	%% SAVING DATA
	%this function gets the edf file from the eyetracker computer and saves it locally. It will print an error if the process is disrupted.
	do_eyetracker_saveclose(settings.eyetracker.dummy_mode, edf_filename);
	
	%End of experiment
end

%% EYETRACKER SETUP FUNCTION
function do_eyetrackersetup(dummy_mode, edf_filename, main_window)
	el = EyelinkInitDefaults(main_window) %collects default settings for this window to use during setup
	
	% Try to initialize the eyetracker. If it fails, throw an error.
	successful_init = EyelinkInit(dummy_mode, 1)
	if ~successful_init
		error('Eyelink Init aborted.')
	end
	clear successful_init
	
	%Display the eyetracker version that's connected
	[v, vs]=Eyelink('GetTrackerVersion');
	fprintf('Running experiment on a ''%s'' tracker.\n', vs );

	% make sure that we get gaze data from the Eyelink
	Eyelink('Command', 'link_sample_data = LEFT,RIGHT,GAZE,AREA');

	% open file to record eye-tracking data to, using the filename we defined at the beginning of the experiment
	Eyelink('Openfile', edf_filename);

	% Calibrate the eye tracker
	EyelinkDoTrackerSetup(el);
	% do a final check of calibration using driftcorrection
	EyelinkDoDriftCorrection(el);
	
end

%% EYETRACKER SAVE AND CLOSE
function do_eyetracker_saveclose(dummy_mode)
	if ~dummy_mode)
		try
		fprintf('Receiving data file ''%s''\n', edf_filename );
				status=Eyelink('ReceiveFile');
				if status > 0
					fprintf('ReceiveFile status %d\n', status);
				end
				if 2==exist(edfFile, 'file')
					fprintf('Data file ''%s'' saved\n', edf_filename);
				end
			catch rdf
				fprintf('Problem receiving data file ''%s''\n', edf_filename );
				rdf;
		end
		
		Eyelink('CloseFile');
		end
	end
end
%% TRIAL DISPLAY FUNCTION
[trial_accuracy, trial_rt] = function RunTrial(trialdata)

	%Code for your trial goes here. This includes displaying pictures, playing sounds, and collecting responses
end 